/**
* \file: CarPlayAudioCodecGst.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPLay
*
* \author: Sudha Kuppusamy/ sudha.kuppusamy@in.bosch.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <gst/app/gstappsrc.h>
#include <gst/app/gstappsink.h>
#include <sys/prctl.h>

#include "APSAudioConverter.h"
#include "CarPlayAudioCodec.h"

#define OPUS_AUDIO_SAMPLE_SIZE   20
#define AUDIO_THREADS_DEFAULT_PRIORITY 61
#define GST_MAIN_LOOP_PRIO_DEFAULT 19

using namespace std;

const char* AudioCodecNames[] =
{
    "beepdec",
    "opusdec",
    "opusenc"
};
const char* AudioCodecNameStrings[] =
{
    "AAC_LC_Decoder",
    "OPUS_Decoder",
    "OPUS_Encoder"
};

static const char* _audioEncAppSrc = "audio_encoder_src";
static const char* _audioEncAppSink = "audio_encoder_sink";

static const char* _audioDecAppSrc = "audio_decoder_src";
static const char* _audioDecAppSink = "audio_decoder_sink";
adit::carplay::IDynamicConfiguration* config;

LOG_IMPORT_CONTEXT(cply_codec);

extern "C" void carplay_InitConfiguration(adit::carplay::IDynamicConfiguration* inConfig)
{
    config = inConfig;
}

static void Set_RealTime_Prio(const char* name, int priority)
{
    if(priority > 0)
    {
        gint err;
        struct sched_param sched_param;

        // get scheduling parameters
        err = sched_getparam(0, &sched_param);
        if(err < 0)
        {
            LOG_ERROR((cply_codec, "CodecError: failed to get parameters with error:%s(%d)", strerror(errno), err));
        }
        else
        {
            // set priority
            sched_param.__sched_priority = priority;

            err = sched_setscheduler(0, SCHED_FIFO, &sched_param);
            if(err < 0)
                LOG_ERROR((cply_codec, "CodecError: failed to set priority for %s with error %s(%d)", name, strerror(errno), err));
        }
    }
}

void* gstMainLoop(void* inData)
{
    if(inData != NULL)
    {
        struct GstreamerPipeline* pipeline = (struct GstreamerPipeline*)inData;

        // set thread name
        prctl(PR_SET_NAME, "CodecGstMainLoop", 0, 0, 0);

        Set_RealTime_Prio("GstMainLoop", config->GetNumber("gstreamer-main-loop_thread-prio", GST_MAIN_LOOP_PRIO_DEFAULT));

        // create g_main_context
        GMainContext *context = g_main_context_new();
        pipeline->loop = g_main_loop_new(context, FALSE);
        if (pipeline->loop == NULL)
        {
            LOG_ERROR((cply_codec, "CodecError: g_main_loop_new failed"));
            return NULL;
        }
        g_main_context_push_thread_default(context);

        g_main_loop_run(pipeline->loop);

        // unref context
        g_main_context_pop_thread_default(context);
        g_main_context_unref(context);
        context = NULL;
    }

    return NULL;
}

GstBusSyncReply audioConverterbusCallBack(GstBus* inBus, GstMessage* inMessage, gpointer inPointer)
{
    (void)inBus;
    (void)inPointer;
//    AudioConverterPrivateRef const me = (AudioConverterPrivateRef)inPointer;

    if(inMessage != NULL)
    {
        switch (GST_MESSAGE_TYPE(inMessage))
        {
            case GST_MESSAGE_EOS:
            {
                //end-of-stream will be sent during flush
                //if(me->pipeline->loop != NULL)
                //{
                //    g_main_loop_quit(me->pipeline->loop);
                //    g_main_loop_unref(me->pipeline->loop);
                //}
                LOGD_DEBUG((cply_codec, "CodecError: gstreamer bus callback EOS received"));
                break;
            }

            case GST_MESSAGE_ERROR:
            {
                gchar* debug;
                GError* error;
                gst_message_parse_error(inMessage, &error, &debug);
                LOG_ERROR((cply_codec, "CodecError: gstreamer bus callback error: %s", error->message));
                g_free(debug);
                g_error_free(error);

                break;
            }

            case GST_MESSAGE_STREAM_STATUS:
            {
                if(config)
                {
                    if(!config->GetNumber("disable-real-time-priority-audio", 0))
                    {
                        GstStreamStatusType type;
                        GstElement* owner = NULL;
                        const GValue* val = NULL;

                        gst_message_parse_stream_status (inMessage, &type, &owner);
                        val = gst_message_get_stream_status_object (inMessage);

                        if(val && type == GST_STREAM_STATUS_TYPE_ENTER)
                        {
                            Set_RealTime_Prio(GST_ELEMENT_NAME(owner), config->GetNumber("audio-threads-real-time-priority", AUDIO_THREADS_DEFAULT_PRIORITY));
                        }
                    }
                }
                break;
            }
            default:
                break;
        }
    }

    return GST_BUS_PASS;
}

static OSStatus audioConverterNewPipeline(AudioConverterPrivateRef codecFormat, AudioCodec AudioDecoderEncoder)
{
    OSStatus err = kNoErr;
    GstElement* pipeline;
    gboolean LinkErr;
    char ConverterPipeline[100];

    /*
     pipeline to be created: gst-launch appsrc ! codec ! appsink
     */

    sprintf(ConverterPipeline, "%s_pipeline", AudioCodecNameStrings[AudioDecoderEncoder]);

    pipeline = gst_pipeline_new(ConverterPipeline);

    if((codecFormat->sourceFormatID == kAudioFormatMPEG4AAC ) || (codecFormat->sourceFormatID == kAudioFormatOpus ))
    {
        //Get appsrc and appsink for decoder
        codecFormat->pipeline->appsrc = gst_element_factory_make("appsrc", _audioDecAppSrc);
        codecFormat->pipeline->appsink = gst_element_factory_make("appsink", _audioDecAppSink);
    }
    else
    {
        //Get appsrc and appsink for encoder
        codecFormat->pipeline->appsrc = gst_element_factory_make("appsrc", _audioEncAppSrc);
        codecFormat->pipeline->appsink = gst_element_factory_make("appsink", _audioEncAppSink);
    }

    if(AudioDecoderEncoder == AAC_LC_Decoder)
    {
        if(config)
        {
            string AACLCCodec = config->GetItem("aaclc-decoder", "");
            codecFormat->pipeline->codec = gst_element_factory_make(AACLCCodec.c_str(), "Audiocodec");
        }
        else
        {
            LOG_ERROR((cply_codec, "Library not loaded or config not initialized"));
        }
    }
    else
    {
        codecFormat->pipeline->codec = gst_element_factory_make(AudioCodecNames[AudioDecoderEncoder], "Audiocodec");
    }

    if(!pipeline || !codecFormat->pipeline->appsrc || !codecFormat->pipeline->appsink
            || !codecFormat->pipeline->codec )
    {
        LOG_ERROR((cply_codec, "CodecError: could not create one of the element for %s pipeline, appsrc %p, appsink: %p, codec: %p", ConverterPipeline, codecFormat->pipeline->appsrc, codecFormat->pipeline->appsink, codecFormat->pipeline->codec));
        err = kUnknownErr;
    }
    else
    {
        // Add all elements to the pipeline
        gst_bin_add_many(GST_BIN(pipeline), codecFormat->pipeline->appsrc, codecFormat->pipeline->codec, codecFormat->pipeline->appsink, NULL);

        //Link all elements together
        LinkErr = gst_element_link_many(codecFormat->pipeline->appsrc, codecFormat->pipeline->codec, codecFormat->pipeline->appsink, NULL);
        if(!LinkErr)
        {
            LOG_ERROR((cply_codec, "CodecError: could not link elements for %s", ConverterPipeline));
            err = kUnknownErr;
        }
    }

    if(err == kNoErr)
    {
        codecFormat->nativeCodecRef = pipeline;
        LOGD_VERBOSE((cply_codec, "%s Codec Pipeline created", AudioCodecNameStrings[AudioDecoderEncoder]));
    }
    else
    {
        gst_object_unref(GST_OBJECT(pipeline));
        gst_object_unref(GST_OBJECT(codecFormat->pipeline->appsrc));
        gst_object_unref(GST_OBJECT(codecFormat->pipeline->appsink));
        gst_object_unref(GST_OBJECT(codecFormat->pipeline->codec));
    }

    return err;
}

static OSStatus audioConverterCreateMainLoopAndBus(AudioConverterPrivateRef codecFormat)
{
    //start g_main_loop thread
    if (0 != pthread_create(&codecFormat->pipeline->mainLoopThreadID, NULL, gstMainLoop, codecFormat->pipeline))
    {
        codecFormat->pipeline->mainLoopThreadID = 0;
        LOG_ERROR((cply_codec, "CodecError: could not create Gstreamer message bus thread"));
        return kUnknownErr;
    }
    // add bus callback
    codecFormat->pipeline->bus = gst_pipeline_get_bus(GST_PIPELINE(codecFormat->nativeCodecRef));
    if (codecFormat->pipeline->bus != NULL)
    {
#if GST_CHECK_VERSION(1,0,0)
        gst_bus_set_sync_handler(codecFormat->pipeline->bus, audioConverterbusCallBack, (void*)codecFormat->pipeline, NULL);
#else
        gst_bus_set_sync_handler(codecFormat->pipeline->bus, audioConverterbusCallBack, (void*)codecFormat->pipeline);
#endif
    }
    else
    {
        LOG_ERROR((cply_codec, "CodecError: gst_pipeline_get_bus failed"));
        return kUnknownErr;
    }

    return kNoErr;
}

static OSStatus audioConverterSetCaps(AudioConverterPrivateRef codecFormat)
{
    GstCaps* srcCaps = NULL;
    GstCaps* sinkCaps = NULL;
    string srccapstr;
    string sinkcapstr;

    switch(codecFormat->sourceFormatID)
    {
        case kAudioFormatMPEG4AAC:
            //caps for appsrc
            srcCaps = gst_caps_new_simple(  "audio/mpeg",
                                            "mpegversion", G_TYPE_INT, 4,
                                            "stream-format", G_TYPE_STRING, "raw",
                                            "framed", G_TYPE_BOOLEAN, TRUE,
#if GST_CHECK_VERSION(1,0,0)
                                            "level", G_TYPE_STRING, "2",
                                            "base-profile", G_TYPE_STRING, "lc",
                                            "profile", G_TYPE_STRING, "lc",
#endif
                                            "rate", G_TYPE_INT, codecFormat->sampleRate,
                                            "channels", G_TYPE_INT, codecFormat->channels,
                                            NULL);

            //caps for appsink
#if GST_CHECK_VERSION(1,0,0)
            sinkCaps = gst_caps_new_simple( "audio/x-raw",
                                            "format", G_TYPE_STRING, "S16LE",
                                            "layout", G_TYPE_STRING, "interleaved",
#else
            sinkCaps = gst_caps_new_simple( "audio/x-raw-int",
                                            "width", G_TYPE_INT, codecFormat->bitsPerChannel,
#endif
                                            "rate", G_TYPE_INT, codecFormat->sampleRate,
                                            "channels", G_TYPE_INT, codecFormat->channels,
                                            NULL);
            break;

            case kAudioFormatOpus:
                        // caps for appsrc
                        srcCaps = gst_caps_new_simple(  "audio/x-opus",
#if GST_CHECK_VERSION(1,0,0)
                                                        "channel-mapping-family", G_TYPE_INT, 0,
                                                        "stream-count", G_TYPE_INT, 1,
                                                        "coupled-count", G_TYPE_INT, 1,
#endif
                                                        "rate", G_TYPE_INT, codecFormat->sampleRate,
                                                        "channels", G_TYPE_INT, codecFormat->channels,
                                                        NULL);
                        //caps for appsink
#if GST_CHECK_VERSION(1,0,0)
                        sinkCaps = gst_caps_new_simple( "audio/x-raw",
#else
                        sinkCaps = gst_caps_new_simple( "audio/x-raw-int",
                                                        "width", G_TYPE_INT, codecFormat->bitsPerChannel,
#endif
                                                        "rate", G_TYPE_INT, codecFormat->sampleRate,
                                                        "channels", G_TYPE_INT, codecFormat->channels,
                                                        NULL);
                    break;

                case kAudioFormatLinearPCM:
                    // caps for appsrc
#if GST_CHECK_VERSION(1,0,0)
                    srcCaps = gst_caps_new_simple( "audio/x-raw",
                                                   "format", G_TYPE_STRING, "S16LE",
                                                   "layout", G_TYPE_STRING, "interleaved",
#else
                    srcCaps = gst_caps_new_simple( "audio/x-raw-int",
                                                   "signed", G_TYPE_BOOLEAN, TRUE,
                                                   "endianness", G_TYPE_INT, 1234,
                                                   "width", G_TYPE_INT, 16,
                                                   "depth", G_TYPE_INT, 16,
#endif
                                                   "rate", G_TYPE_INT, codecFormat->sampleRate,
                                                   "channels", G_TYPE_INT, codecFormat->channels,
                                                   NULL);


            //caps for appsink
            sinkCaps = gst_caps_new_simple( "audio/x-opus",
                                            "rate", G_TYPE_INT, codecFormat->sampleRate,
                                            "channels", G_TYPE_INT, codecFormat->channels,
                                            NULL);
            break;
    }

    if (srcCaps != NULL)
    {
        g_object_set(G_OBJECT(codecFormat->pipeline->appsrc),
                            "block",        TRUE,
                            "is-live",      TRUE,
                            "format",       GST_FORMAT_TIME,
                            "min-percent",  100,
                            "caps",         srcCaps,
                            NULL);
    }
    else
    {
        LOG_ERROR((cply_codec, "CodecError: could not create caps for appsrc"));
        return kUnknownErr;
    }


    if(sinkCaps != NULL)
    {
        g_object_set(G_OBJECT(codecFormat->pipeline->appsink),
                                "sync",            FALSE,
                                "caps",            sinkCaps,
                                NULL);
    }
    else
    {
        LOG_ERROR((cply_codec, "CodecError: could not create caps for appsink"));
        return kUnknownErr;
    }

    return kNoErr;
}

static OSStatus audioConverterStartPipeline(AudioConverterPrivateRef inConverter)
{
    OSStatus err = kNoErr;
    GstStateChangeReturn ret;

    ret = gst_element_set_state (GST_ELEMENT(inConverter->nativeCodecRef), GST_STATE_PLAYING);
    if(ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((cply_codec, "CodecError: Error in setting the pipeline to PLAYING state"));
        audioConverterDestroyGstPipeline(inConverter);
        err = kUnknownErr;
    }

    return err;
}

OSStatus audioConverterCreateGstPipeline(AudioConverterPrivateRef codecFormat, AudioCodec AudioDecoderEncoder)
{
    OSStatus err;

    // Initialize pipeline structure
    codecFormat->pipeline = (struct GstreamerPipeline*)calloc(1, sizeof(*codecFormat->pipeline));
    if(codecFormat->pipeline == NULL)
    {
        LOG_ERROR((cply_codec, "CodecError: Memory allocation failed for pipeline structure"));
        return kNoMemoryErr;
    }

    /* create a new pipeline */
    err = audioConverterNewPipeline(codecFormat, AudioDecoderEncoder);

    if(err == kNoErr)
    {
        err = audioConverterCreateMainLoopAndBus(codecFormat);
        if(err == kNoErr)
        {
            err = audioConverterSetCaps(codecFormat);
            if(err == kNoErr)
            {
                // set the pipeline to PLAYING state
                err = audioConverterStartPipeline(codecFormat);
            }
        }
    }

    if((err == kNoErr) && (AudioDecoderEncoder == OPUS_Encoder))
    {
        if(codecFormat->pipeline->codec != NULL)
        {
            // Setting below properties recommended by Apple
            g_object_set(G_OBJECT(codecFormat->pipeline->codec),
#if GST_CHECK_VERSION(1,0,0)
                        "audio-type",               2048,     // OPUS_APPLICATION_VOIP
                        "bitrate-type",             2,        // OPUS_SET_VBR_CONSTRAINT
#else
                        "audio",                    FALSE,    // OPUS_APPLICATION_VOIP
                        "cbr",                      FALSE,    // OPUS_SET_VBR
                        "constrained-vbr",          TRUE,     // OPUS_SET_VBR_CONSTRAINT
#endif
                        "complexity",               10,
                        "inband-fec",               FALSE,
                        "dtx",                      FALSE,
                        "packet-loss-percentage",   FALSE,
                        "bandwidth",                -1000,
                        NULL);
        }

        // Initialize local buffer for buffering
        int frame_size = (codecFormat->sampleRate*OPUS_AUDIO_SAMPLE_SIZE)/1000;
        codecFormat->pcmBuffer = (uint8_t*)calloc(1, frame_size*codecFormat->channels*sizeof(int16_t));
        if(codecFormat->pcmBuffer != NULL)
            codecFormat->pcmBufferPos = 0;
        else
            err = kNoMemoryErr;
    }

    if(err != kNoErr)
    {
        audioConverterDestroyGstPipeline(codecFormat);
    }
    else
    {
        codecFormat->pipelineCreated= true;

        LOG_INFO((cply_codec, "%s Codec pipeline state set to PLAYING", AudioCodecNameStrings[AudioDecoderEncoder]));
    }

    return err;
}

void audioConverterDestroyGstPipeline(AudioConverterPrivateRef codecFormat)
{
    GstStateChangeReturn ret;

    if(codecFormat->pipeline != NULL)
    {
        // stop g_main_loop thread
        if(codecFormat->pipeline->mainLoopThreadID != 0)
        {
            if(codecFormat->pipeline->loop != NULL)
            {
                g_main_loop_quit(codecFormat->pipeline->loop);
                g_main_loop_unref(codecFormat->pipeline->loop);
                codecFormat->pipeline->loop = NULL;
            }
            pthread_join(codecFormat->pipeline->mainLoopThreadID, NULL);
            codecFormat->pipeline->mainLoopThreadID = 0;
        }

        if (codecFormat->pipeline->bus != NULL)
        {
           gst_object_unref(codecFormat->pipeline->bus);
           codecFormat->pipeline->bus = NULL;
        }

        free(codecFormat->pipeline);
        codecFormat->pipeline = NULL;
        codecFormat->pipelineCreated = false;
    }

    ret = gst_element_set_state (GST_ELEMENT(codecFormat->nativeCodecRef), GST_STATE_NULL);

    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((cply_codec, "failed to set_state GST_STATE_NULL for codec pipeline"));
    }

    if (codecFormat->nativeCodecRef != NULL)
    {
       gst_object_unref(codecFormat->nativeCodecRef);

       codecFormat->nativeCodecRef = NULL;
    }

    if(codecFormat->pcmBuffer != NULL)
    {
        free(codecFormat->pcmBuffer);
        codecFormat->pcmBuffer = NULL;
    }
}

static OSStatus _AudioConverterPullAndCopyOutputData(AudioConverterPrivateRef inConverter, AudioBufferList* outOutputData)
{
    OSStatus err = kNoErr;
    GstBuffer* receiveBuffer;


#if GST_CHECK_VERSION(1,0,0)
    GstSample* samples  = gst_app_sink_pull_sample(GST_APP_SINK(inConverter->pipeline->appsink));
    receiveBuffer = gst_sample_get_buffer (samples);
    if(samples==NULL || receiveBuffer == NULL)
    {
        LOG_ERROR((cply_codec, "codecError:gst_app_sink_pull_sample returned NULL"));
        return kUnknownErr;
    }

    GstMapInfo info;
    if(receiveBuffer && gst_buffer_map(receiveBuffer, &info, GST_MAP_READ))
    {
        if(info.data != NULL)
        {
            if((outOutputData->mBuffers[0].mData != NULL) && (info.size <= outOutputData->mBuffers[ 0 ].mDataByteSize))
            {
                memcpy(outOutputData->mBuffers[ 0 ].mData, info.data, info.size);
                outOutputData->mBuffers[ 0 ].mDataByteSize = info.size;
                gst_buffer_unmap(receiveBuffer, &info);
            }
            else
            {
                LOG_ERROR((cply_codec, "gst buffer_map failed"));
                err = kUnknownErr;
            }
        }

#else
    receiveBuffer = gst_app_sink_pull_buffer(GST_APP_SINK(inConverter->pipeline->appsink));
    if(receiveBuffer && receiveBuffer->data)
    {
        if((outOutputData->mBuffers[0].mData != NULL) && (GST_BUFFER_SIZE(receiveBuffer) <= outOutputData->mBuffers[ 0 ].mDataByteSize))
        {
            memcpy(outOutputData->mBuffers[ 0 ].mData, GST_BUFFER_DATA(receiveBuffer), GST_BUFFER_SIZE(receiveBuffer));
            outOutputData->mBuffers[ 0 ].mDataByteSize = GST_BUFFER_SIZE(receiveBuffer);
        }
#endif
        else
        {
            LOG_ERROR((cply_codec, "CodecError: AudioConverter Output buffer is NULL or size is less"));
            err = kNoMemoryErr;
        }
    }
    else
    {
        LOG_ERROR((cply_codec, "CodecError: gst_app_sink_pull_buffer() returned NULL"));
        err = kUnknownErr;
    }

#if GST_CHECK_VERSION(1,0,0)
    if(samples != NULL)
        gst_sample_unref(samples);
#else
    // unref receive buffer
    if(receiveBuffer != NULL)
        gst_buffer_unref(receiveBuffer);
#endif

    return err;
}

OSStatus _AudioConverterSendEOS(AudioConverterPrivateRef inConverter)
{
    GstFlowReturn   err       = GST_FLOW_OK;
    err = gst_app_src_end_of_stream(GST_APP_SRC_CAST(inConverter->pipeline->appsrc));
    if(err != GST_FLOW_OK)
        return kUnknownErr;
    else
        return kNoErr;
}

OSStatus _AudioConverterConvertAudioData(AudioConverterPrivateRef inConverter, void* inData, uint32_t inDataSize, AudioBufferList* outOutputData)
{
    OSStatus        retErr;
    GstFlowReturn   err       = GST_FLOW_OK;
    GstBuffer*      sendBuffer;

    if(inData == NULL)
    {
        LOG_ERROR((cply_codec, "CodecError: Input buffer is empty. Cannot perform Audio conversion"));
        return kUnknownErr;
    }

    // create Gstreamer buffer
    sendBuffer = gst_buffer_new_and_alloc(inDataSize);
#if GST_CHECK_VERSION(1,0,0)
    GstMapInfo info;
    if(sendBuffer && gst_buffer_map (sendBuffer, &info, GST_MAP_WRITE))
    {
        if(info.data != NULL)
        {
            memcpy (info.data, (uint8_t*)inData, inDataSize);
            gst_buffer_unmap (sendBuffer, &info);
        }
        else
        {
            LOG_ERROR((cply_codec, "CodecError: gst buffer_map failed"));
            return kUnknownErr;
        }
    }
    else
    {
        LOG_ERROR((cply_codec, "CodecError: gst buffer NULL"));
        return kNoMemoryErr;
    }
#else
    gst_buffer_set_data(sendBuffer, (guint8*)inData, inDataSize);
#endif
    err = gst_app_src_push_buffer(GST_APP_SRC_CAST(inConverter->pipeline->appsrc), sendBuffer);
    if(err == GST_FLOW_OK)
    {
        /* Pull the buffer from appsink, this blocks until
         * data is available in appsink */
        retErr = _AudioConverterPullAndCopyOutputData(inConverter, outOutputData);
    }
    else
    {
        LOG_ERROR((cply_codec, "CodecError: Error in queuing the data, %d", err));
        retErr = kUnknownErr;
    }

   return retErr;
}

